' ********************************************* SSB RECEIVER **************************************
'
' SSB receiver for 3.0 MHz to 30 MHz
'
' Processor ATMega328 on Arduino NANO
' Signal source SI5351A module.
' IF 9 MHz
' BFO frequencies 8996.6 MHz and 8999.6 MHz


$regfile = "M328def.dat"                                    ' Chip description
$hwstack = 100                                              ' make sure stacks big enough
$swstack = 100
$framesize = 100

$crystal = 16000000                                         ' 16MHz clock,
Config Com1 = 38400 , Synchrone = 0 , Parity = None , Stopbits = 1 , Databits = 8 , Clockpol = 0
                                       ' push button switch on encoder
Config Adc = Single , Prescaler = Auto , Reference = Avcc

Config Lcd = 16 * 2


Cursor On Noblink
Config Lcdpin = Pin , Db4 = Portd.4 , Db5 = Portd.5 , Db6 = Portd.6 , Db7 = Portd.7 , E = Portb.0 , Rs = Portb.1
Waitms 50
Config Com1 = 38400 , Synchrone = 0 , Parity = None , Stopbits = 1 , Databits = 8 , Clockpol = 0

Mute Alias Portb.5
Enca Alias Pind.2                                           'rotary encoder input A
Encb Alias Pinb.2                                           'rotary encoder input B

Ddrb = &B10_0011                                            ' Port B0..1 LCD control
Portb = &B00_0111                                           '

Ddrc = &B0000_0010                                          ' Port C0 .. 1 capacitor select
Portc = &B0000_1000                                         '

Ddrd = &B1111_0000                                          ' Port D4..7 LCD data
Portd = &B0000_0000                                         ' All others are inputs


 ' **************************************** VARIABLE DEFINITIONS ***********************************
'
' *************************************************** BIT *****************************************


Dim Step_bit As Bit
Dim R_flag As Bit                                           'frequency below 500kHz required Rdivider of 128
Dim Divby4 As Bit                                           'flags when frequency is above 150MHz
Dim Frequency_up As Bit
Dim Frequency_down As Bit
Dim Vfoab As Bit
Dim Save_flag As Bit
Dim Direction_bit As Bit                                    ' encoder direction bit
Dim Sideband As Bit
Dim Sideband_previous As Bit
Dim En_bit As Bit
Dim Direction As Bit

' *************************************************** BYTE ****************************************

Dim Temp As Byte                                            'temporary variable used in sendfreq and sbarplot routines
Dim Rdivider As Byte                                        'Normally 1 unless frequency<1MHz then 1,2,4,8,16,32,64 or 128 only
Dim Rbyte As Byte                                           'used as temp store before final data load to si5351
Dim Cap_value As Byte
Dim Readings As Byte                                        ' used in FOR .. NEXT
Dim Encoder_direction As Byte                               ' Bit 0 toggles direction
Dim Index As Byte
Dim Previous_mhz As Byte
Dim En_byte As Byte
Dim Loop_count As Byte
Dim Xtal_a As Byte
Dim Xtal_b As Byte
Dim Xtal_c As Byte
Dim Xtal_d As Byte

' **************************************************** WORD ***************************************

Dim Msfactor As Word
Dim Temp_1 As Word
Dim Rssi As Word                                            ' signal strength
Dim Rssi_previous As Word
Dim Mhz_start As Word
Dim Band_number As Word
Dim Band_start As Word
Dim Previous_band As Word
Dim Previous_hz As Word
Dim Vtune As Word
Dim Previous_vtune As Word
Dim Adc_value As Word

' **************************************************** DWORD **************************************

Dim Afrequency As Dword                                     'four byte unsigned variable for VFO A
Dim Afrequency_max As Dword
Dim Afrequency_min As Dword
Dim Vfo_frequency As Dword
Dim Bfrequency As Dword                                     'and VFO B
Dim Gfreq As Dword
Dim Ifoffset As Dword
Dim Afactor As Dword
Dim Bfactor As Dword
Dim Vdp1 As Dword                                           '18-bit variable for P1
Dim Cfactor As Dword
Dim Mhz_display As Dword
Dim Khz_display As Dword
Dim Hz_display As Dword
Dim Temp_2 As Dword
Dim Step_size As Dword                                      ' linear step size close to resonance
Dim Afreq_previous As Dword
Dim Temp1 As Dword
Dim Previous_khz As Dword
Dim Delta As Dword
Dim Khz_value As Dword
Dim Half_mhz As Dword
Dim Xtal As Dword                                           ' crystal frequency on 5351
Dim Xtal_1 As Dword
Dim Xtal_2 As Dword
Dim Xtal_3 As Dword
Dim Xtal_4 As Dword
Dim Term1 As Dword
Dim Term2 As Dword

' *************************************************************************************************

'si5351 parameters for variable divider

Dim Vdp11 As Byte At Vdp1 Overlay                           'mirrored variables for dds calculation
Dim Vdp12 As Byte At Vdp1 + 1 Overlay
Dim Vdp13 As Byte At Vdp1 + 2 Overlay
Dim Vdp14 As Byte At Vdp1 + 3 Overlay                       'most significant byte

Dim Vdp2 As Dword                                           '20-bit variable for P2
Dim Vdp21 As Byte At Vdp2 Overlay                           'mirrored variables for dds calculation
Dim Vdp22 As Byte At Vdp2 + 1 Overlay
Dim Vdp23 As Byte At Vdp2 + 2 Overlay
Dim Vdp24 As Byte At Vdp2 + 3 Overlay                       'most significant byte

Dim Vdp3 As Dword                                           '20-bit variable for P3
Dim Vdp31 As Byte At Vdp3 Overlay                           'mirrored variables for dds calculation
Dim Vdp32 As Byte At Vdp3 + 1 Overlay
Dim Vdp33 As Byte At Vdp3 + 2 Overlay
Dim Vdp34 As Byte At Vdp3 + 3 Overlay                       'most significant byte


Dim Regaddr As Byte , Regdata As Byte                       'for i2c calls
Dim Siaddr1 As Byte , Siaddr2 As Byte , Siaddr3 As Byte     'for writing to si5351 CLK0 and CLK1

Dim Oscillator As Byte                                      'pointer to currently selected oscillator

Cr Alias &H0D
Band_switch Alias Portc.1

Enable Int0
Enable Int1
Config Int1 = Falling
Config Int0 = Falling
Enable Interrupts                                           'enable all interrupts
On Int0 Rotary_encoder
On Int1 Frequency_step

' Declare Constants
Const Cvalue = 1048575                                      'used in MSYNTH calculations
'Xtal = 25000000
'Gosub Store_xtal


' Read EEPROM locations 1-4 for XTAL frequency
  Gosub Xtal_frequency                                      ' returns value in XTAL
  Print "Xtal = " ; Xtal ; Chr(cr);
  Adc_value = Getadc(7)
  If Adc_value > 1000 Then Gosub Calibrate


'********************************* Declare Subroutines  **********************************************

Declare Sub Si5351init                                      'initialises si5351
Declare Sub Calculate                                       'converts current VFO or BFO freq to data for si5351
Declare Sub Sendpllfreq                                     'sends VFO or BFO/CIO freq data to si5351 PLL
Declare Sub Sendmsynthfreq                                  'sends VFO or BFO/CIO freq data to si5351 MSYNTH
Declare Sub Oscrefresh                                      'updates oscillator settings
Declare Sub Siout(sireg As Byte , Sidata As Byte)           'for i2c write to si5351

Config Sda = Portc.4
Config Scl = Portc.5
Config Adc = Single , Prescaler = Auto , Reference = Avcc   'starts automatically





' initialise i2c here...
I2cinit


Mute = 1



'initialise some variables

Previous_mhz = 0
Previous_khz = 0

 '  Gosub Band_pot
 '  Gosub Ssb_switch
'and initialise the flags

Reset Vfoab                                                 'select VFO A to begin at power up
Reset R_flag                                                'status of R-divider stage - set for freqs below 500kHz
Reset Divby4                                                'flag set when VFO freq>150MHz

'initialise the si5351a with CLK0 (VFO) and CLK2 (BFO/CIO) outputs
'and initialise the flags

Reset Vfoab                                                 'select VFO A to begin at power up
Reset R_flag                                                'status of R-divider stage - set for freqs below 500kHz
'Reset Divby4                                      'flag set when VFO freq>150MHz

Splash_screen:
   Cls
   Lcd "  SSB Receiver  "
   Locate 2 , 1
   Lcd "Ver.5.2 July 2025"
   Wait 1
   Cls

   Print
   Print "SSB Receiver " ;
   Print "Ver.5.2 July 2025" ; Chr(cr);

' initialise the si5351a with CLK0 (VFO) and CLK2 (BFO/CIO) outputs


Step_size = 2

Si5351init

                                                  'initialises si5351
Gosub Initial_band_pot                                      ' starting band
Gosub Initial_ssb

Gfreq = Afrequency + Ifoffset
Oscillator = 1
'Print "Signal = " ; Afrequency ; ; " VFO = " ; Gfreq ;
Oscrefresh

Gfreq = Ifoffset
Oscillator = 3                                              'VFO is OUT#0
Print " BFO = " ; Ifoffset ; " Hz" ; Chr(cr);               'VFO is OUT#0
Oscrefresh
Locate 1 , 12
Lcd "MHz"
Cap_value = 0
Gosub Capacitor


Dim Encoder_interrupt As Bit
Dim Encoder_byte As Bit
Band_start = 6
Waitms 500                                                  'interrupt encoder
Mute = 0

' ****************************************** MAIN *************************************************
'
' Test frequency adjustment, USB/LSB switch, BAND potentiometer, RSSI level


Do

Main:
      If Frequency_up = 1 Then Gosub Increase_frequency
      If Frequency_down = 1 Then Gosub Decrease_frequency
      If Step_bit = 1 Then Gosub Change_step
      Gosub Ssb_switch
      Gosub Rssi_measure
      Gosub Band_pot
      Loop



' ********************************************* STEP SIZE *****************************************
'
' Toggle through step sizes of 10, 100, 1k, 10k, 100k, 1000k Hz


Frequency_step:
    Waitms 10
    If Pind.3 = 0 Then Set Step_bit
    Print Pind.3;
    Return


Change_step:
   Step_bit = 0
   Incr Step_size
   If Step_size = 6 Then Step_size = 0
   Print Step_size ; Chr(cr);
   Gosub Locate_step
   Return

Locate_step:
   If Step_size = 0 Then Locate 1 , 9
   If Step_size = 1 Then Locate 1 , 8
   If Step_size = 2 Then Locate 1 , 6
   If Step_size = 3 Then Locate 1 , 5
   If Step_size = 4 Then Locate 1 , 4
   If Step_size = 5 Then Locate 1 , 2
   Return



' ********************************* ROTARY ENCODER INT0 *******************************************
'
' Rotary encoder steps frequency and capacitor value up or down
' Uses encoder_direction to suit encoder


Rotary_encoder:

 Waitms 2
   En_byte = Pinb
   En_byte = En_byte And &B0000_0100
   Waitms 10
   Direction = Pinc.3
   If Direction = 0 Then Gosub Normal
   If Direction = 1 Then Gosub Reverse
   Return

Normal:
   If En_byte = 0 Then Frequency_up = 1
   If En_byte = 4 Then Frequency_down = 1
   Return

Reverse:
   If En_byte = 4 Then Frequency_up = 1
   If En_byte = 0 Then Frequency_down = 1
Return

' ************************************** FREQUENCY CHANGE *****************************************

Increase_frequency:
   Frequency_up = 0
   Frequency_down = 0                                       ' zero flag
 '  Mute = 1
 '  Step_bit = 0

   Afrequency_max = 30000000
   If Step_size = 0 Then Afrequency = Afrequency + 10
   If Step_size = 1 Then Afrequency = Afrequency + 100
   If Step_size = 2 Then Afrequency = Afrequency + 1000
   If Step_size = 3 Then Afrequency = Afrequency + 10000
   If Step_size = 4 Then Afrequency = Afrequency + 100000
   If Step_size = 5 Then Afrequency = Afrequency + 1000000
   If Afrequency > Afrequency_max Then Afrequency = 3000000
   If Afrequency = 8900000 Then Afrequency = Afrequency + 200000
   Gosub Display_frequency
   Gosub Set_vfo                                            'VFO is OUT#0

   Return


Decrease_frequency:
   Mute = 1
   Afrequency_min = 3000000
   Frequency_down = 0
   Frequency_up = 0
   If Step_size = 0 Then Afrequency = Afrequency - 10
   If Step_size = 1 Then Afrequency = Afrequency - 100
   If Step_size = 2 Then Afrequency = Afrequency - 1000
   If Step_size = 3 Then Afrequency = Afrequency - 10000
   If Step_size = 4 Then Afrequency = Afrequency - 100000
   If Step_size = 5 Then Afrequency = Afrequency - 1000000
   If Afrequency = 9100000 Then Afrequency = Afrequency - 200000
   If Afrequency < Afrequency_min Then Afrequency = 30000000
   Gosub Display_frequency
   Gosub Set_vfo

    Return



' *************************************** DISPLAY FREQUENCY ***************************************

' Display frequency om line 1
' e.g. 14.123.450 MHz




Display_frequency:
   Mute = 1

   Khz_value = Afrequency / 1000
   If Afrequency => 10000000 Then Band_switch = 1
   If Afrequency < 10000000 Then Band_switch = 0
   Locate 1 , 1
   Mhz_display = Afrequency / 1000000                       ' calculate MHz
   Half_mhz = Afrequency / 500000
   If Mhz_display = Previous_mhz Then Goto Df_5
   Previous_mhz = Mhz_display

   If Mhz_display < 10 Then Lcd " " ;                       'Leading Blank
   Lcd Mhz_display ; ".";
Df_5:
   Locate 1 , 4
   Khz_display = Afrequency Mod 1000000
   Temp1 = Khz_display
   Khz_display = Khz_display / 1000                         ' just three digits
   If Khz_display = Previous_khz Then Goto Df_10
    Previous_khz = Khz_display
 '
Df_10:
  ' Locate 1 , 8
   If Khz_display = 0 Then Lcd "000";
   If Khz_display = 0 Then Goto Dh_10
   If Khz_display < 10 Then Lcd "00" ; Khz_display ;
   If Khz_display < 10 Then Goto Dh_10
   If Khz_display < 100 Then Lcd "0" ; Khz_display ;
   If Khz_display => 100 Then Lcd Khz_display
Dh_10:
   Lcd ".";
   Hz_display = Temp1 Mod 1000
   If Hz_display = 0 Then Lcd "000";
   If Hz_display = 0 Then Goto Dh_exit
   If Hz_display < 10 Then Lcd "00" ; Hz_display ;
   If Hz_display < 10 Then Goto Dh_exit
   If Hz_display < 100 Then Lcd "0" ; Hz_display ;
   If Hz_display => 100 Then Lcd Hz_display
Dh_exit:
   If Band_number < 2 Then Gosub Calculate_low              ' low band capacitance
   If Band_number => 2 Then Gosub Calculate_high            ' high band capacitance
   Gosub Locate_step
   Mute = 0

   Return



' ------------------------------------------------------------------------------------
'
'
' Value of resonant capacitor from 3 to 10 MHz
' Capacitors are 4.7, 10, 22, 47, 120, 330
' Varicap plus strays ~15 pF
'
'   Half MHz   Capacitance    Value
'     6       380       10_0110
'     7       280       01_1111
'     8       213       01_1100
'     9       168       01_0110
'     10      136       01_0000
'     11      113       01_0010
'     12       95       00_1111
'     13       80       00_1100
'     14       69       00_1010
'     15       60       00_0111
'     16       52       00_0111
'     17       47       00_0111_
'     18       41       00_0110
'     19       37       00_0101
'     20       33       00_0101


Calculate_low:
   If Half_mhz = 6 Then Cap_value = &B10_1111
   If Half_mhz = 7 Then Cap_value = &B01_1111
   If Half_mhz = 8 Then Cap_value = &B01_0110
   If Half_mhz = 9 Then Cap_value = &B01_0010
   If Half_mhz = 10 Then Cap_value = &B01_1100
   If Half_mhz = 11 Then Cap_value = &B00_0110
   If Half_mhz = 12 Then Cap_value = &B00_1111
   If Half_mhz = 13 Then Cap_value = &B00_1100
   If Half_mhz = 14 Then Cap_value = &B00_1010
   If Half_mhz = 15 Then Cap_value = &B00_0111
   If Half_mhz = 16 Then Cap_value = &B00_0111
   If Half_mhz = 17 Then Cap_value = &B00_0111
   If Half_mhz = 18 Then Cap_value = &B00_0110
   If Half_mhz = 19 Then Cap_value = &B00_0101
   If Half_mhz = 20 Then Cap_value = &B00_0101

   Gosub Capacitor
   Return


' -----------------------------------------------------------------------------------
'
' Value of resonant capacitor from 10 to 30 MHz
' Capacitors are 4.7, 10, 22, 47, 120, 330
' Varicap plus strays ~15 pF
'
'    MHz   Capacitance    Value
'     10       233       01_1111
'     11       190       01_1010
'     12       162       01_0110
'     13       140       01_0010
'     14       120       01_0000
'     15       104       00_1111
'     16        90       00_1110
'     17        80       00_1100
'     18        70       00_1010
'     19        62       00_1001
'     20        58       00_1001
'     21        53       00_0111_
'     22        49       00_0110
'     23        45       00_0101
'     24        40       00_0101
'     25        37       00_0100
'     26        33       00_0100
'     27        31       00_0011
'     28        30       00_0011
'     29        28       00_0011
'     30        24       00_0010


 Calculate_high:
   If Mhz_display = 10 Then Cap_value = &B01_1111
   If Mhz_display = 11 Then Cap_value = &B01_1000
   If Mhz_display = 12 Then Cap_value = &B01_0100
   If Mhz_display = 13 Then Cap_value = &B01_0000
   If Mhz_display = 14 Then Cap_value = &B00_1111
   If Mhz_display = 15 Then Cap_value = &B00_1111
   If Mhz_display = 16 Then Cap_value = &B00_1110
   If Mhz_display = 17 Then Cap_value = &B00_1100
   If Mhz_display = 18 Then Cap_value = &B00_1010
   If Mhz_display = 19 Then Cap_value = &B00_1000
   If Mhz_display = 20 Then Cap_value = &B00_1000
   If Mhz_display = 21 Then Cap_value = &B00_0110
   If Mhz_display = 22 Then Cap_value = &B00_0110
   If Mhz_display = 23 Then Cap_value = &B00_0100
   If Mhz_display = 24 Then Cap_value = &B00_0100
   If Mhz_display = 25 Then Cap_value = &B00_0011
   If Mhz_display = 26 Then Cap_value = &B00_0011
   If Mhz_display = 27 Then Cap_value = &B00_0001
   If Mhz_display = 28 Then Cap_value = &B00_0001
   If Mhz_display = 29 Then Cap_value = &B00_0001
   If Mhz_display = 30 Then Cap_value = &B00_0000
   Gosub Capacitor
   Return


' ******************************************** RSSI MEASURE ***************************************
'
' Measure RSSI on ADC6 and display on line 2 .
' Measures voltage from 0 to 4.0V which corresponds to counts of about 0 to 820
' Divide that into 10 is 33 counts per blob.

Rssi_measure:
   Rssi = Getadc(6)
   Rssi = Rssi / 82                                         ' scale as 0 to 10
  ' Print "RSSI = " ; Rssi ; Chr(cr);
   If Rssi = Rssi_previous Then Goto Rm_exit
   Rssi_previous = Rssi
   Locate 2 , 7
   Lcd "          "
   If Rssi > 10 Then Goto Rm_exit
   Locate 2 , 7
   For Index = 1 To Rssi
   Lcd Chr(255);                                            'Chr(255);
   Next

Rm_exit:
   Gosub Locate_step
    Return



' ******************************************* SSB switch ******************************************
'
' If sideband = 1 then LSB else USB

Ssb_switch:
   Sideband = Pinc.0                                        ' read LSB/USB switch
   If Sideband_previous = Sideband Then Goto Sw_exit
   Sideband_previous = Sideband
   Locate 2 , 1
   If Sideband = 0 Then Lcd "USB"
   If Sideband = 1 Then Lcd "LSB"
   If Sideband = 0 Then Ifoffset = 8999600                  ' USB
   If Sideband = 1 Then Ifoffset = 8996600                  ' LSB
   Print "BFO = " ; Ifoffset ; "Hz" ; Chr(cr);
   Gfreq = Ifoffset
   Oscillator = 3
   Oscrefresh
Sw_exit:
   Return

Initial_ssb:
   Sideband = Pinc.0                                        ' read LSB/USB switch
   Locate 2 , 1
   If Sideband = 0 Then Lcd "USB"
   If Sideband = 1 Then Lcd "LSB"
   If Sideband = 0 Then Ifoffset = 8999600                  ' USB
   If Sideband = 1 Then Ifoffset = 8996600                  ' LSB
   Gfreq = Ifoffset
   Print "BFO = " ; Ifoffset ; "Hz" ; Chr(cr);
   Oscillator = 3
   Oscrefresh
   Return



' **************************************** BAND START *********************************************
'
' Rotating the pot brings the frequency inside each band by about 100 kHz.
' Bands are:
' 3500-3800 kHz
' 7000-7300 kHz
' 10.0-10.15 MHz
' 14.0-14.35 MHz
' 21.0-21.50 MHz
' 28.0-29.7 MHz

' Divide up 0 - 1023 into 6 bands
'
' The VFO operates above the frequency.

Band_pot:
   Band_start = Getadc(7)
   Band_number = Band_start / 170                           ' 6 count segments
   If Band_number = Previous_band Then Goto Bp_exit
   Previous_band = Band_number
   Gosub Select_band
   Gosub Set_vfo
Bp_exit:
   Return


Set_vfo:

   Mute = 1
   Waitms 10
   Vfo_frequency = Afrequency + Ifoffset
   Print "VFO = " ; Vfo_frequency ; " Hz" ; Chr(cr);
   Gfreq = Vfo_frequency
   Oscillator = 1                                           'VFO is OUT#0
   Oscrefresh
   Waitms 30
   Mute = 0                                                 ' turn on audio
   Return

Initial_band_pot:
   Band_start = Getadc(7)
   Band_number = Band_start / 170                           ' 7 count segments
   Gosub Select_band
   Gosub Locate_step
   Return


Select_band:
   If Band_number = 0 Then Gosub 3500_khz
   If Band_number = 1 Then Gosub 7000_khz
   If Band_number = 2 Then Gosub 10_mhz
   If Band_number = 3 Then Gosub 14_mhz
   If Band_number = 4 Then Gosub 21_mhz
   If Band_number = 5 Then Gosub 28_mhz
   Gosub Set_vfo
   Gosub Display_frequency
   Gosub Locate_step
 '  Print "VFO = " ; Vfo_frequency ; " Hz" ; Chr(cr);
   Return






3500_khz:
    Afrequency = 3600000
    Band_switch = 0

    Return

7000_khz:
    Afrequency = 7100000
    Band_switch = 0

    Return

10_mhz:
    Afrequency = 10000000
    Band_switch = 1                                         ' 1uH transformer coil

    Return

14_mhz:
    Afrequency = 14100000
    Band_switch = 1                                         ' 1uH transformer coil

    Return

21_mhz:
    Afrequency = 21100000
    Band_switch = 1

    Return

28_mhz:
    Afrequency = 28100000
    Band_switch = 1                                         ' 1uH transformer coil

    Return




' *************************************** PCF8574 OUTPUT ******************************************
'
' Write a single byte to slave address 20H
' Entry is capacitance value

Capacitor:
    Mute = 1
    I2cstart
    I2cwbyte &H40                                           'i2c address of PCF8574
    Waitus 1                                                'suggested delay from forum
    I2cwbyte Cap_value
    Waitus 1
    I2cstop
    Waitms 50
    Mute = 0
    Gosub Locate_step
Return

' ************************************** CALCULATE XTAL ***********************************
'
' XTAL frequency is in EEPROM locations 1-4
' returns 32 bit value in XTAL.
' Starting values are 1, 125, 120, 64


Xtal_frequency:

   Readeeprom Xtal_a , 1                                    ' Most significant
   Readeeprom Xtal_b , 2
   Readeeprom Xtal_c , 3
   Readeeprom Xtal_d , 4                                    ' least significant
'   Print "Xtal read=" ; Xtal_a ; " " ; Xtal_b ; " " ; Xtal_c ; " " ; Xtal_d ; Chr(cr);
   Xtal_1 = Xtal_a                                          ' transfer 8 bit to 32 bit
   Xtal_2 = Xtal_b
   Xtal_3 = Xtal_c
   Xtal_4 = Xtal_d
   Xtal_1 = Xtal_1 * 16777216
   Xtal_2 = Xtal_2 * 65536
   Xtal_3 = Xtal_3 * 256
   Xtal = Xtal_1
   Xtal = Xtal + Xtal_2
   Xtal = Xtal + Xtal_3
   Xtal = Xtal + Xtal_4
   Return

' stores new value of XTAL into EEPROM

Store_xtal:
   Xtal_1 = Xtal / 16777216
   Xtal_2 = Xtal Mod 16777216
   Temp1 = Xtal_2 Mod 65536
   Xtal_2 = Xtal_2 / 65536
   Xtal_3 = Temp1 / 256
   Xtal_4 = Temp1 Mod 256
   Writeeeprom Xtal_1 , 1
   Writeeeprom Xtal_2 , 2
   Writeeeprom Xtal_3 , 3
   Writeeeprom Xtal_4 , 4
   Print "Xtal stored=" ; Xtal_1 ; " " ; Xtal_2 ; " " ; Xtal_3 ; " " ; Xtal_4 ; Chr(cr);
   Return

' ***********************************  CALIBRATE **********************************
'
' Calibrate frequency accuracy
'
' Put up message on LCD

Calibrate:
   Mute = 1
   Waitms 250                                               ' wait for switch release
   Gfreq = 10000000
   Oscillator = 1                                           ' OUT#1
   Oscrefresh
   Cls
   Locate 1 , 1
   Lcd " Freq Calibrate "
   Locate 2 , 1
   Lcd Xtal                                                 ' current xtal frequency
Cal_10:
   If Frequency_up = 1 Then Gosub Increase_xtal
   If Frequency_down = 1 Then Gosub Decrease_xtal
   Locate 2 , 1
   Lcd Xtal
   Adc_value = Getadc(7)
   Waitms 20
   If Adc_value > 1000 Then Goto Cal_10                     ' loop for more change
   Print "Cal done" ; Chr(cr);
   Gosub Store_xtal
   Mute = 0
   Cls

   Return

 Increase_xtal:
   Locate 2 , 6
   Frequency_up = 0
   Xtal = Xtal + 10
   Gfreq = 10000000
   Oscillator = 1                                           ' OUT#1
   Oscrefresh
   Print "XU=" ; Xtal ; Chr(cr);
   Return


 Decrease_xtal:
   Locate 2 , 6
   Frequency_down = 0
   Xtal = Xtal - 10
   Gfreq = 10000000
   Oscillator = 1                                           ' OUT#1
   Oscrefresh
   Print "XD=" ; Xtal ; Chr(cr);
   Return


'Subroutines...
'======================================================================
'**********************************************************************
' subroutine to initialise the si5351 chip via I2C
'**********************************************************************

Sub Si5351init

Regaddr = 183
Regdata = 192
Call Siout(regaddr , Regdata)                               'set xtal load cap to 10pF
Regaddr = 3
Regdata = 0
Call Siout(regaddr , Regdata)                               'enable all outputs
Regaddr = 15
Regdata = 0
Call Siout(regaddr , Regdata)                               'select xtal as PLL input source
Regaddr = 16
Regdata = 15                                                'PLLA is source, 8mA output
Call Siout(regaddr , Regdata)                               'configure clock 0 control register
Regaddr = 17
Regdata = 47                                                'PLLB is source, 8mA output
Call Siout(regaddr , Regdata)                               'configure clock 1 control register
Regaddr = 18
Regdata = 47                                                'PLLB is source, 8mA output
Call Siout(regaddr , Regdata)                               'configure clock 2 control register
Regaddr = 177
Regdata = &B10100000                                        'reset PLLA and PLLB                                             '
Call Siout(regaddr , Regdata)
Waitms 10                                                   'wait 10ms just in case

End Sub


'-------------------------------------------------------------------------------
'
' update oscillator frequency routine
'
' PLLA is always used by VFO osc1 (and by osc2 when Quadrature mode is selected)
' PLLB is always used by BFO/CIO osc3 only
' Entry assumes gfreq is holding current AFrequency or BFrequency
' or BFO/CIO frequency (at startup)

Sub Oscrefresh

      Select Case Oscillator                                'depending on the currently selected oscillator

         Case 1 :                                           'set CLK0 freq with f1 - gfreq holds value used for si5351

            Siaddr1 = 26                                    'point to CLK0 base register addresses    (PLLA)
            Siaddr2 = 42                                    'Msynth0
            Siaddr3 = 16                                    'Control0

         Case 2 :                                           'set CLK1 freq

            Siaddr1 = 26                                    'point to CLK1 base register addresses    (PLLA)
            Siaddr2 = 50                                    'Msynth1
            Siaddr3 = 17                                    'Control1

         Case 3 :                                           'set CLK2 freq - gfreq holds value used for si5351

            Siaddr1 = 34                                    'point to CLK2 base register addresses   (PLLB)
            Siaddr2 = 58                                    'Msynth2
            Siaddr3 = 18                                    'Control2

      End Select

      Reset Divby4                                          'reset >150MHz and <500kHz flags before test
      Reset R_flag

      If Gfreq < 500000 Then
         Gfreq = Gfreq * 128
         R_flag = 1
      End If

      Select Case Gfreq                                     'Multisynth integer divider ratio lookup
         Case Is > 150000000 :
            Msfactor = 4
            Set Divby4
         Case Is > 75000000 : Msfactor = 6
         Case Is > 56250000 : Msfactor = 8
         Case Is > 25000000 : Msfactor = 18
         Case Is > 10220000 : Msfactor = 44
         Case Is > 4500000 : Msfactor = 100
         Case Is > 3499000 : Msfactor = 126                 'to squeeze the last drop from quadrature
         Case Is > 2000000 : Msfactor = 220
         Case Is > 800000 : Msfactor = 550
         Case Else : Msfactor = 1800
      End Select

      Cfactor = Msfactor
      Cfactor = Cfactor * Gfreq

      Afactor = Cfactor \ Xtal                              'calculate factors for variable PLL(A or B) assuming a fixed MSYNTH of 4 or 6-1800 based on lookup
      Bfactor = Cfactor Mod Xtal
      Cfactor = Xtal

      While Cfactor > Cvalue
         Shift Bfactor , Right                              'repeated divide by 2 until cfactor < si5351a register limit of 1048575
         Shift Cfactor , Right                              'SHIFT also works with DWORD variables although not documented in bascom
      Wend

      Calculate                                             'and calculate PLL parameters

                                                            'MSynth# and related pointers are selected earlier depending on osc#
      Sendpllfreq                                           'and configure the PLL registers

      If Divby4 = 0 Then
         Vdp1 = Msfactor                                    'and calculate vdp1 here quickly for for MSYNTH write
         Shift Vdp1 , Left , 7                              'vdp1*128  (fast multiply by shifting)
         Vdp1 = Vdp1 - 512
      Else
         Vdp1 = 0                                           'and set vdp1=0 for MSYNTH write when cfactor=4
      End If
      Vdp2 = 0                                              'msynth set to div by 4 or 6-1800 so skip 'calculate' and just load parameters into MSx
      Vdp3 = 1                                              'vdp1 was set above

      Sendmsynthfreq                                        'and configure the selected MSynth registers

End Sub

'*************************************************************************************************
' calculate si5351a data bytes for currently selected frequency to send to si5351a PLL or Msynth
'*************************************************************************************************
Sub Calculate


If R_flag = 1 Then
   Rdivider = 128
   R_flag = 0                                               'reset for next time
Else
   Rdivider = 1                                             'initialise r
End If

Select Case Rdivider                                        'determine value of R44 bits [6:4]
      Case 1 : Rbyte = 0                                    '000 0000
 '     Case 2 : Rbyte = 16                         '001 0000
 '     Case 4 : Rbyte = 32                         '010 0000
 '     Case 8 : Rbyte = 48                         '011 0000
 '     Case 16 : Rbyte = 64                        '100 0000
 '     Case 32 : Rbyte = 80                        '101 0000
 '     Case 64 : Rbyte = 96                        '110 0000
 '     Case 128 : Rbyte = 112                      '111 0000
End Select

Shift Bfactor , Left , 7                                    'vdp2 = 128*b  (fast multiply by shifting)
Vdp2 = Bfactor
Vdp3 = Vdp2 \ Cfactor                                       'use vdp3 as temp store for (128*b)\c
Shift Afactor , Left , 7                                    'vdp1 = 128*a  (fast multiply by shifting)
Vdp1 = Afactor
Vdp1 = Vdp1 + Vdp3
Vdp1 = Vdp1 - 512

Vdp3 = Vdp3 * Cfactor
Vdp2 = Vdp2 - Vdp3

Vdp3 = Cfactor

'calculations are complete so now build and send the required output data bytes by calling sendPLLfreq and/or sendMSYNTHfreq  (About 20 bytes sent per frequency)

End Sub


'**********************************************************************************************************
' send converted freq data to si5351 PLL A/B
'
' vdp11-vdp14 are bytes comprising vdp1 where vdp14 is msb    (Variable Divide _ Parameter1)
' vdp21-vdp24 are bytes comprising vdp2
' vdp31-vdp34 are bytes comprising vdp3
'
' PLL is fixed at 875MHz for gfreq < 109MHz (i.e. vdp1=3968, vdp2=0, vdp3=1) (Associated MSynths are variable)
' PLL is variable for gfreq > 109MHz (Associated MSynths are fixed)
'
'**********************************************************************************************************
Sub Sendpllfreq

Regaddr = Siaddr1                                           'points to R26 or R34 to load PLLA or PLLB respectively

Call Siout(regaddr , Vdp32)                                 'R26=vdp3[15:8]
Incr Regaddr
Call Siout(regaddr , Vdp31)                                 'R27=vdp3[7:0]
Incr Regaddr
Call Siout(regaddr , Vdp13)                                 'R28=vdp1[17:16] in R28[1:0] with R28[7:2]=0
Incr Regaddr
Call Siout(regaddr , Vdp12)                                 'R29=vdp1[15:8]
Incr Regaddr
Call Siout(regaddr , Vdp11)                                 'R30=vdp1[7:0]
Temp = Vdp33                                                'temp[3:0]=vdp3[19:16]
Swap Temp                                                   'temp[7:4]=vdp3[19:16]
Temp = Temp Or Vdp23                                        'so temp[7:4]=vdp3[19:16] and temp[3:0]=vdp2[19:16]
Incr Regaddr
Call Siout(regaddr , Temp)                                  'R31=(vdp3[19:16],vdp2[19:16])
Incr Regaddr
Call Siout(regaddr , Vdp22)                                 'R32=vdp2[15:8]
Incr Regaddr
Call Siout(regaddr , Vdp21)                                 'R33=vdp2[7:0]

End Sub


'**********************************************************************************************************
' send converted freq data to si5351 Msynth MS0, MS1 or MS2
'
' vdp11-vdp14 are bytes comprising vdp1 where vdp14 is msb    (Variable Divide _ Parameter1)
' vdp21-vdp24 are bytes comprising vdp2
' vdp31-vdp34 are bytes comprising vdp3
'
' MSynths are variable for gfreq<109MHz (Associated PLL is fixed at 875MHz)
' MSynths are fixed (div by 4 or 6) when gfreq > 109MHz (Associated PLL is variable)
'
'**********************************************************************************************************

Sub Sendmsynthfreq

Regaddr = Siaddr2                                           'points to R42, R50 or R58 to load MS0, MS1 or MS2 respectively

Call Siout(regaddr , Vdp32)                                 'R42=MS0_P3[15:8]
Incr Regaddr
Call Siout(regaddr , Vdp31)                                 'R43=MS0_P3[7:0]
Temp = Rbyte
Temp = Temp Or Vdp13
If Divby4 = 1 Then                                          'MSYNTHx should be set to divby4
   Temp = Temp Or &B00001100                                'set MSx_DIVBY4[1:0] if flag was set by calculation
End If
Incr Regaddr
Call Siout(regaddr , Temp)                                  'R44=(0,R[6:4],MSx_DIVBY4[1:0],MS0_P1[17:16])
Incr Regaddr
Call Siout(regaddr , Vdp12)                                 'R45=MS0_P1[15:8]
Incr Regaddr
Call Siout(regaddr , Vdp11)                                 'R46=MS0_P1[7:0]
Temp = Vdp33                                                'temp[3:0]=vdp3[19:16]
Swap Temp                                                   'temp[7:4]=vdp3[19:16]
Temp = Temp Or Vdp23                                        'so temp[7:4]=vdp3[19:16] and temp[3:0]=vdp2[19:16]
Incr Regaddr
Call Siout(regaddr , Temp)                                  'R47=(vdp3[19:16],vdp2[19:16])
Incr Regaddr
Call Siout(regaddr , Vdp22)                                 'R48=vdp2[15:8]
Incr Regaddr
Call Siout(regaddr , Vdp21)                                 'R49=vdp2[7:0]


Regaddr = Siaddr3                                           'points to R16, R17 or R18 for MS0, MS1 or MS2 respectively

If Divby4 = 1 Then
   Select Case Oscillator                                   'select PLLA or PLLB as source for MSx (b5=0 or 1 resp) and configure MSx for integer divide mode (b6=1)
      Case 1 : Temp = &B01001101                            'PLLA
      Case 2 : Temp = &B01001101                            'PLLA
      Case 3 : Temp = &B01101101                            'PLLB
   End Select
Else
   Select Case Oscillator                                   'select PLLA or PLLB as source for MSx (b5=0 or 1 resp) and configure MSx for fractional divide mode (b5=0)
      Case 1 : Temp = &B00001101                            'PLLA
      Case 2 : Temp = &B00001101                            'PLLA
      Case 3 : Temp = &B00101101                            'PLLB
   End Select
End If

Call Siout(regaddr , Temp)                                  'send data to si5351a

If Oscillator = 2 Then
   Regaddr = 166                                            'set phase shift if oscillator=2
   Temp = Msfactor                                          'phase shift byte value
   Call Siout(regaddr , Temp)
End If

'


'Regaddr = 3
'Regdata = 0
'Call Siout(regaddr , Regdata)                               'enable all outputs

End Sub


'**********************************************************************
' i2c transmission to si5351
'**********************************************************************
Sub Siout(sireg As Byte , Sidata As Byte)
'subroutine sends databyte to register_nbr at i2c address 192 (si5351 register read is from address 194)

   'Write a single byte (slave address 192, register sireg, value sibyte)
    I2cstart
    I2cwbyte 192                                            'i2c address of si5351a
    Waitus 1                                                'suggested delay from forum
    I2cwbyte Sireg
    Waitus 1
    I2cwbyte Sidata
    Waitus 1
    I2cstop

End Subtms 50                                               'for i2c to settle and for lcd RC reset time



' ******************************** INITIAL EEPROM DATA **************************
'
' Initial frequency of 25,000,000 into locations 1 to 4 of EEPROM

25mhz_data:
$eeprom
Data 0 , 1 , 125 , 120 , 64